# (c) Copyright 2022. CodeWeavers, Inc.

import traceback

from gi.repository import Gtk

import bottlecollection
import bottlemanagement
import bottlequery

import cxdiag
import cxguitools
import cxproduct
import cxutils

import pyop
import ratingdialog

from cxutils import cxgettext as _


def archive_bottle(bottle, window):
    file_chooser = Gtk.FileChooserDialog(
        title=_("Choose an Archive Name and Location"),
        transient_for=window,
        action=Gtk.FileChooserAction.SAVE)

    cxguitools.add_filters(file_chooser, cxguitools.FILTERS_CXARCHIVES)
    file_chooser.add_button(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL)
    button = file_chooser.add_button(Gtk.STOCK_SAVE, Gtk.ResponseType.OK)
    button.get_style_context().add_class('suggested-action')
    file_chooser.set_current_name(bottle.name + ".cxarchive")

    file_chooser.set_do_overwrite_confirmation(True)

    cxguitools.set_default_extension(file_chooser, 'cxarchive')

    if file_chooser.run() == Gtk.ResponseType.OK:
        filename = file_chooser.get_filename()

        def callback(operation):
            if not operation.exit_status[0]:
                cxguitools.CXMessageDlg(
                    primary=_('An error occurred while archiving %s') % operation.bottle.name,
                    secondary=operation.exit_status[1], message_type=Gtk.MessageType.ERROR,
                    parent=window)

        archiveOp = ArchiveBottleOperation(bottle, filename, callback)
        pyop.sharedOperationQueue.enqueue(archiveOp)

    file_chooser.destroy()


def delete_bottle(bottle, window):
    primary_text = _("Are you sure you want to delete the '%s' bottle?") % bottle.name
    secondary_text = _("Deleting this bottle will permanently erase all Windows applications contained in it, as well as any documents stored in its C: drive.")

    buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
               (Gtk.STOCK_DELETE, Gtk.ResponseType.OK, "destructive-action"))

    def delete_bottle_response(response):
        if response != Gtk.ResponseType.OK:
            return

        delete_bottle_op = DeleteBottleOperation(bottle)
        pyop.sharedOperationQueue.enqueue(delete_bottle_op)

        config = cxproduct.get_config()
        ask_for_ratings = config['CrossOver'].get('AskForRatings', '1') != '0'
        if bottle.appid and ask_for_ratings:
            ratingdialog.RatingController(bottle.name, bottle.appid, window, True, False)

    cxguitools.CXMessageDlg(primary=primary_text, secondary=secondary_text, button_array=buttons,
                            response_function=delete_bottle_response,
                            message_type=Gtk.MessageType.WARNING, parent=window)


def open_c_drive(bottle):
    if not bottle.is_usable:
        return

    environ = bottlequery.get_win_environ(bottle.name, ('SystemDrive',))
    drive = bottlequery.expand_win_string(environ, '%SystemDrive%')
    drive = bottlequery.get_native_path(bottle.name, drive)

    args = ['xdg-open', drive]
    try:
        cxutils.run(args, background=True)
    except: # pylint: disable=W0702
        traceback.print_exc()


def quit_bottle(bottle, force, delegate=None):
    quit_bottle_op = QuitBottleOperation([bottle.name], force, delegate)
    pyop.sharedOperationQueue.enqueue(quit_bottle_op)


def set_dxvk_state(bottle, state, delegate=None):
    if state != (bottle.is_dxvk_enabled_state == bottle.STATUS_DXVK_ENABLED):
        set_dxvk_op = SetDxvkOperation(bottle, state, delegate)
        pyop.sharedOperationQueue.enqueue(set_dxvk_op)


def set_esync_state(bottle, state, window, delegate=None):
    if state != (bottle.is_esync_enabled_state == bottle.STATUS_ESYNC_ENABLED):
        if state:
            primary_text = _("Some Windows applications will not work correctly with ESync enabled.")
            buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                       (_("Reboot Bottle and Enable ESync"), Gtk.ResponseType.OK))
        else:
            primary_text = None
            buttons = ((Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL),
                       (_("Reboot Bottle and Disable ESync"), Gtk.ResponseType.OK))

        secondary_text = _("Changing this setting will reboot the bottle and cause any running applications in the bottle to quit. Are you sure you want to do this?")

        def toggle_esync_response(response):
            if response == Gtk.ResponseType.OK:
                set_esync_op = SetESyncOperation(bottle, state, delegate)
                pyop.sharedOperationQueue.enqueue(set_esync_op)
            else:
                bottle.bottle_changed()

        cxguitools.CXMessageDlg(primary=primary_text, secondary=secondary_text, button_array=buttons,
                                response_function=toggle_esync_response,
                                message_type=Gtk.MessageType.WARNING, parent=window)


def set_highres_state(bottle, state, delegate=None):
    if state != (bottle.is_high_resolution_enabled_state == bottle.STATUS_HIGHRES_ENABLED):
        set_highres_op = SetHighResOperation(
            bottle, state, int(float(cxdiag.get(None).properties.get('display.dpi', 0))), delegate)
        pyop.sharedOperationQueue.enqueue(set_highres_op)


class ArchiveBottleOperation(pyop.PythonOperation):

    def __init__(self, bottle, archive_path, callback):
        pyop.PythonOperation.__init__(self)
        self.exit_status = None
        self.bottle = bottle
        self.archive_path = archive_path
        self.callback = callback

    def __unicode__(self):
        return "ArchiveBottleOperation for " + self.bottle.name

    def enqueued(self):
        self.bottle.add_status_override(self.bottle.STATUS_ARCHIVING)
        pyop.PythonOperation.enqueued(self)

    def main(self):
        self.exit_status = bottlemanagement.archive_bottle(self.bottle.name, self.archive_path)

    def finish(self):
        self.bottle.remove_status_override(self.bottle.STATUS_ARCHIVING)
        self.callback(self)
        pyop.PythonOperation.finish(self)


class DeleteBottleOperation(pyop.PythonOperation):

    def __init__(self, bottle):
        pyop.PythonOperation.__init__(self)
        self.bottle_name = bottle.name
        self.bottle = bottle
        self.deleted = False

    def __unicode__(self):
        return "DeleteBottleOperation for " + self.bottle_name

    def enqueued(self):
        self.bottle.add_status_override(self.bottle.STATUS_DELETING)
        pyop.PythonOperation.enqueued(self)

    def main(self):
        ret = bottlemanagement.delete_bottle(self.bottle_name, self.bottle.is_managed)
        self.deleted = ret[0]

    def finish(self):
        if self.deleted:
            bottlecollection.sharedCollection().refresh()
        else:
            self.bottle.remove_status_override(self.bottle.STATUS_DELETING)

        pyop.PythonOperation.finish(self)


class QuitBottleOperation(pyop.PythonOperation):

    def __init__(self, inBottleList, inForce=False, inFinishDelegate=None):
        pyop.PythonOperation.__init__(self)
        self.bottleList = [cxutils.expect_unicode(x) for x in inBottleList]
        self.force = inForce
        self.finishDelegate = inFinishDelegate
        self.succeeded = True

    def __unicode__(self):
        return " ".join(["QuitBottleOperation for"] + self.bottleList)

    def main(self):
        for bottleName in self.bottleList:
            bottleObj = bottlecollection.sharedCollection().bottleObject(bottleName)
            if self.force:
                if not bottleObj.force_quit():
                    self.succeeded = False
            else:
                if not bottleObj.quit():
                    self.succeeded = False
            bottleObj.cancelShutdown()

    def finish(self):
        if self.finishDelegate:
            self.finishDelegate.op_finished(self)

        pyop.PythonOperation.finish(self)


class SetDxvkOperation(pyop.PythonOperation):

    def __init__(self, bottle, enabled, delegate):
        pyop.PythonOperation.__init__(self)
        self.enabled = enabled
        self.bottle = bottle
        self.bottle_name = bottle.name
        self.delegate = delegate

    def __unicode__(self):
        return "SetDxvkOperation for " + self.bottle_name

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        if self.enabled:
            self.bottle.enable_dxvk()
        else:
            self.bottle.disable_dxvk()

    def finish(self):
        if self.delegate:
            self.delegate.op_finished(self)

        pyop.PythonOperation.finish(self)


class SetESyncOperation(pyop.PythonOperation):

    def __init__(self, bottle, enabled, delegate):
        pyop.PythonOperation.__init__(self)
        self.enabled = enabled
        self.bottle = bottle
        self.bottle_name = bottle.name
        self.delegate = delegate

    def __unicode__(self):
        return "SetESyncOperation for " + self.bottle_name

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        if self.bottle.quit():
            if self.enabled:
                self.bottle.enable_esync()
            else:
                self.bottle.disable_esync()

    def finish(self):
        if self.delegate:
            self.delegate.op_finished(self)

        pyop.PythonOperation.finish(self)


class SetHighResOperation(pyop.PythonOperation):

    def __init__(self, bottle, enabled, dpi, delegate):
        pyop.PythonOperation.__init__(self)
        self.enabled = enabled
        self.bottle = bottle
        self.bottle_name = bottle.name
        self.dpi = dpi
        self.delegate = delegate

    def __unicode__(self):
        return "SetHighResOperation for " + self.bottle_name

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        if self.enabled:
            self.bottle.enable_high_resolution(self.dpi)
        else:
            self.bottle.disable_high_resolution()

    def finish(self):
        if self.delegate:
            self.delegate.op_finished(self)

        pyop.PythonOperation.finish(self)
