# (c) Copyright 2009-2015. CodeWeavers, Inc.

import os

from gi.repository import GLib
from gi.repository import Gtk

import cxutils
import cxobservable
import distversion

import cxguitools
import pyop

import demoutils
import webtoken

# for localization
from cxutils import cxgettext as _

# These are the observable events
DIALOG_CLOSED = 'dialog_closed'


#
# Dialog instance
#

class RegisterDialogController(cxobservable.Object):

    # This is the list of observable events
    observable_events = frozenset((DIALOG_CLOSED,))

    def __init__(self):
        cxobservable.Object.__init__(self)

        # Import widget modules so Gtk.Builder() finds them
        import cxhtmltextview # pylint: disable=W0612,W0611
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("cxregister"))
        self.xml.connect_signals(self)

        icons = cxguitools.get_std_icon_list('cxregister')
        self.xml.get_object("RegisterDialog").set_icon_list(icons)

        explanationText = _("""<body>
                             <p>This is a time limited trial version of CrossOver.</p>
                             <p>Visit <a href="%(url)s">%(visibleurl)s</a> to purchase a license to keep using CrossOver.</p>
                             <p>If you have already purchased a license, enter your codeweavers.com account email and password to unlock CrossOver.</p>
                          </body>""") % {'url': distversion.DEMO_URL,
                                         'visibleurl': distversion.DEMO_URL_VISIBLE,
                                         'productname': distversion.PRODUCT_NAME}

        self.xml.get_object("ExplanationTextView").display_html_safe(explanationText)

        normal_unlocking_text = "<body><p>%s</p></body>" % cxutils.html_escape(_("If you purchased directly from codeweavers.com, you do not have an activation code and should proceed to 'normal unlocking'."))
        self.xml.get_object("NormalUnlockingTextView").display_html(normal_unlocking_text)

        reset_password_markup = '<small><a href="https://www.codeweavers.com/login/lostpassword/">%s</a></small>' % \
            cxutils.html_escape(_("Forgot Password?"))
        self.xml.get_object("ResetPasswordLink").set_markup(reset_password_markup)

        self.xml.get_object("RegisterButton").set_can_default(True)

        self.normalProgbar = self.xml.get_object("RegistrationProgbar")
        self.normalProgbar.set_pulse_step(0.05)
        self.normalProgbar.hide()

        self.serialProgbar = self.xml.get_object("SerialRegistrationProgbar")
        self.serialProgbar.set_pulse_step(0.05)
        self.serialProgbar.hide()

        self.progBar = self.normalProgbar

        self.normalWindow = self.xml.get_object('RegisterDialog')
        self.serialCodeWindow = self.xml.get_object('SerialCodeRegisterWindow')
        self.mainWindow = self.normalWindow

        self.serialCodeWindow.connect('destroy', self.activation_cancel)

        self.animateEvent = None
        self.update_sensitivities(None)

        self.xml.get_object("EmailEntry").grab_focus()

    def update_sensitivities(self, _caller):
        if self.xml.get_object("PasswordEntry").get_text() and self.xml.get_object("EmailEntry").get_text():
            self.xml.get_object("RegisterButton").set_sensitive(True)
        else:
            self.xml.get_object("RegisterButton").set_sensitive(False)
            self.xml.get_object("RegisterButton").grab_default()

    def sensitivities_at_start_registration(self):
        self.xml.get_object("RegisterButton").set_sensitive(False)
        self.xml.get_object("CancelButton").set_sensitive(False)
        self.xml.get_object("PasswordEntry").set_sensitive(False)
        self.xml.get_object("EmailEntry").set_sensitive(False)
        self.xml.get_object("serialEmailEntry").set_sensitive(False)
        self.xml.get_object("serialEmailEntry2").set_sensitive(False)
        self.xml.get_object("serialPasswordEntry").set_sensitive(False)
        self.xml.get_object("serialPasswordEntry2").set_sensitive(False)

        self.animateEvent = GLib.timeout_add(100, self.progbar_pulse)

        self.progBar.show()

    def sensitivities_after_registration(self):
        self.xml.get_object("RegisterButton").set_sensitive(True)
        self.xml.get_object("CancelButton").set_sensitive(True)
        self.xml.get_object("PasswordEntry").set_sensitive(True)
        self.xml.get_object("EmailEntry").set_sensitive(True)
        self.xml.get_object("serialEmailEntry").set_sensitive(True)
        self.xml.get_object("serialEmailEntry2").set_sensitive(True)
        self.xml.get_object("serialPasswordEntry").set_sensitive(True)
        self.xml.get_object("serialPasswordEntry2").set_sensitive(True)

        GLib.source_remove(self.animateEvent)

        self.progBar.hide()
        self.update_sensitivities(None)

    def progbar_pulse(self):
        self.progBar.pulse()
        return True

    def normal_register_clicked(self, _caller):
        self.sensitivities_at_start_registration()
        registerOp = RegisterOperation(self, self.xml.get_object("EmailEntry").get_text(),
                                       self.xml.get_object("PasswordEntry").get_text())
        pyop.sharedOperationQueue.enqueue(registerOp)

    def serial_register_finished(self, success, errorString):

        if success:
            registerOp = RegisterOperation(self,
                                           self.xml.get_object("serialEmailEntry").get_text(),
                                           self.xml.get_object("serialPasswordEntry").get_text())
            pyop.sharedOperationQueue.enqueue(registerOp)
        else:
            self.sensitivities_after_registration()
            cxguitools.CXMessageDlg(primary=_("Registration failed"),
                                    secondary=errorString,
                                    buttons=Gtk.ButtonsType.OK,
                                    parent=self.mainWindow,
                                    message_type=Gtk.MessageType.ERROR)

    def normal_register_finished(self, success, errorString):
        self.sensitivities_after_registration()

        if success:
            cxguitools.CXMessageDlg(_("Registration successful."), buttons=Gtk.ButtonsType.OK,
                                    response_function=self.quit_requested, parent=self.mainWindow,
                                    message_type=Gtk.MessageType.INFO)
        else:
            cxguitools.CXMessageDlg(primary=_("Registration failed"), secondary=errorString,
                                    buttons=Gtk.ButtonsType.OK, parent=self.mainWindow,
                                    message_type=Gtk.MessageType.ERROR)

    def get_email_format_error(self, email, email2):
        # pylint: disable=R0201
        if email.count("@") != 1:
            return _("Please enter a valid email address.\n(e.g. myusername@myemailprovider.com)")

        if email != email2:
            return _("The provided email addresses do not match")

        return ""

    def get_password_format_error(self, password, password2):
        # pylint: disable=R0201
        if password != password2:
            return _("The provided passwords do not match")

        return ""

    def get_serial_format_error(self, dash_code):
        # pylint: disable=R0201
        code = dash_code.replace("-", "")

        if len(code) == 19:
            return ""

        if len(code) < 19:
            # pylint: disable=W0511
            return _("This serial code is not valid - it is too short.\n(Example: XXX-XXXX-XXXX-XXXXXXXX).")
        # pylint: disable=W0511
        return _("This serial code is not valid - it is too long.\n(Example: XXX-XXXX-XXXX-XXXXXXXX).")

    def format_serial_code(self, dash_code):
        # pylint: disable=R0201
        code = dash_code.replace("-", "")
        return code[:3] + "-" + code[3:7] + "-" + code[7:11] + "-" + code[11:]

    def serial_register_clicked(self, _caller):
        emailFormatError = self.get_email_format_error(self.xml.get_object("serialEmailEntry").get_text(),
                                                       self.xml.get_object("serialEmailEntry2").get_text())
        passwordFormatError = self.get_password_format_error(self.xml.get_object("serialPasswordEntry").get_text(),
                                                             self.xml.get_object("serialPasswordEntry2").get_text())
        serialCodeFormatError = self.get_serial_format_error(self.xml.get_object("serialCodeEntry").get_text())

        self.xml.get_object("emailAddressErrorLabel").set_text(emailFormatError)
        self.xml.get_object("passwordErrorLabel").set_text(passwordFormatError)
        self.xml.get_object("serialCodeErrorLabel").set_text(serialCodeFormatError)

        if not (emailFormatError == "" and passwordFormatError == "" and serialCodeFormatError == ""):
            return

        self.xml.get_object("serialCodeEntry").set_text(
            self.format_serial_code(self.xml.get_object("serialCodeEntry").get_text()))

        self.sensitivities_at_start_registration()
        registerOp = SerialCodeAccountCreationOperation(self,
                                                        self.xml.get_object("serialEmailEntry").get_text(),
                                                        self.xml.get_object("serialPasswordEntry").get_text(),
                                                        self.xml.get_object("serialNameEntry").get_text(),
                                                        self.xml.get_object("serialCodeEntry").get_text())
        pyop.sharedOperationQueue.enqueue(registerOp)

    def show_serial_register_window(self, _caller):
        self.xml.get_object("emailAddressErrorLabel").set_text("")
        self.xml.get_object("passwordErrorLabel").set_text("")
        self.xml.get_object("serialCodeErrorLabel").set_text("")

        self.mainWindow = self.serialCodeWindow
        self.progBar = self.serialProgbar
        self.normalWindow.set_sensitive(False)
        self.present()

    def show_normal_register_window(self, _caller):
        self.mainWindow = self.normalWindow
        self.progBar = self.normalProgbar
        self.serialCodeWindow.hide()
        self.present()

    def present(self):
        self.mainWindow.set_sensitive(True)
        self.mainWindow.present()

    def quit_requested(self, _caller):
        self.mainWindow.destroy()
        self.emit_event(DIALOG_CLOSED)
        cxguitools.toplevel_quit()

    def activation_cancel(self, _caller):
        self.normalWindow.destroy()
        self.quit_requested(None)


#
# Global dialog management
#

DIALOG = None

def _dialog_closed(_dialog, _event, _data):
    # pylint: disable=W0603
    global DIALOG
    DIALOG = None

def open_or_show(close_when_finished=False):
    license_file = os.path.join(cxutils.CX_ROOT, "etc", "license.txt")
    sig_file = os.path.join(cxutils.CX_ROOT, "etc", "license.sha256")
    if not os.path.exists(sig_file):
        sig_file = os.path.join(cxutils.CX_ROOT, "etc", "license.sig")
    (isdemo, username, _date, _licenseid, _revoked) = demoutils.demo_status(license_file, sig_file)

    if isdemo:
        # pylint: disable=W0603
        global DIALOG
        if DIALOG is None:
            DIALOG = RegisterDialogController()
            DIALOG.add_observer(DIALOG_CLOSED, _dialog_closed)
        else:
            DIALOG.present()
        return DIALOG
    if username is not None:
        msg = _("This copy of %(product)s was unlocked by:\n\n%(name)s\n\nIt is fully enabled. Thank you for your support!") % {'product': distversion.PRODUCT_NAME, 'name': username}
    else:
        msg = _("This copy of %s is already fully enabled.\n\nThank you for your support!") % distversion.PRODUCT_NAME
    if close_when_finished:
        import sys
        my_response_func = sys.exit
    else:
        my_response_func = None
    cxguitools.CXMessageDlg(msg, response_function=my_response_func, buttons=Gtk.ButtonsType.OK, message_type=Gtk.MessageType.INFO)
    return None


#
# Operations
#

class SerialCodeAccountCreationOperation(pyop.PythonOperation):

    def __init__(self, inRegisterController, inLogin, inPassword,
                 inUserName, inSerialCode):
        pyop.PythonOperation.__init__(self)
        self.registerController = inRegisterController
        self.login = inLogin
        self.password = inPassword
        self.success = False
        self.userName = inUserName
        self.serialCode = inSerialCode
        self.errorString = None

    def __unicode__(self):
        return "SerialCodeAccountCreationOperation with username " + self.login

    def main(self):
        self.success = False

        self.errorString = demoutils.create_acct_apply_scode(self.login,
                                                             self.password,
                                                             self.userName,
                                                             self.serialCode)
        if self.errorString is None:
            self.success = True

    def finish(self):
        self.registerController.serial_register_finished(self.success,
                                                         self.errorString)
        pyop.PythonOperation.finish(self)



class RegisterOperation(pyop.PythonOperation):

    def __init__(self, inRegisterController, inLogin, inPassword):
        pyop.PythonOperation.__init__(self)
        self.registerController = inRegisterController
        self.login = inLogin
        self.password = inPassword
        self.success = False
        self.errorString = None

    def __unicode__(self):
        return "RegisterOperation with username " + self.login

    def main(self):
        self.success, self.errorString = demoutils.register(self.login, self.password)

        if not self.success:
            return

        # Also create a token that can be used for the REST API. No error handling
        # here since this is not essential, and we already know that the login
        # is valid.
        webtoken.create_token(self.login, self.password)

    def finish(self):
        self.registerController.normal_register_finished(self.success,
                                                         self.errorString)
        pyop.PythonOperation.finish(self)
