# (c) Copyright 2009-2022. CodeWeavers, Inc.

from gi.repository import Gdk, Gtk

import c4profiles
import cxfixes
import cxguitools
import cxproduct
import cxutils
import distversion
import installtask
import loggingoptions

from cxutils import cxgettext as _


class InstallOptionsDialogController:

    def __init__(self, install_task, transient_for):
        self.install_task = install_task

        # Import widget modules so Gtk.Builder() finds them
        import cxhtmltextview # pylint: disable=W0612,W0611

        self.xml = Gtk.Builder()
        self.xml.set_translation_domain('crossover')
        self.xml.add_from_file(cxguitools.get_ui_path('installoptionsdialog'))
        self.xml.connect_signals(self)

        self.xml.get_object('InstallOptionsDialog').set_transient_for(transient_for)
        self.xml.get_object('InstallUnixPackagesButton').set_label(
            _("The following %(platform)s packages may be needed by this application and will be installed system-wide:") %
            {'platform': distversion.PLATFORM})

        global_config = cxproduct.get_config()
        if global_config['OfficeSetup'].get('ApplyCxfixes', '1') != '1':
            self.xml.get_object('InstallUnixPackagesButton').set_active(False)

        self.dependency_list_model = Gtk.ListStore(object, str, str) # appid, name, reason

        column = Gtk.TreeViewColumn(_('Name'), Gtk.CellRendererText(), text=1)
        self.xml.get_object('DependencyList').append_column(column)
        column = Gtk.TreeViewColumn(_('Reason'), Gtk.CellRendererText(), text=2)
        self.xml.get_object('DependencyList').append_column(column)
        self.xml.get_object('DependencyList').set_model(self.dependency_list_model)

        self.add_dependencies_list_model = Gtk.ListStore(object, str, str) # appid, name, sort key
        self.add_dependencies_list_model.set_sort_column_id(2, Gtk.SortType.ASCENDING)

        self.xml.get_object('AddDependencyList').append_column(Gtk.TreeViewColumn(
            'Name', Gtk.CellRendererText(), text=1))
        self.xml.get_object('AddDependencyList').get_selection().set_mode(
            Gtk.SelectionMode.MULTIPLE)
        self.xml.get_object('AddDependencyList').get_selection().connect(
            'changed', self.on_AddDependencyList_selection_changed)

        self.update_cxfixes_packages()
        self.update_dependencies()

        self.language_list_model = Gtk.ListStore(str, object) # name, langid
        self.language_list_model.set_sort_func(1, self._language_list_cmp)
        self.language_list_model.set_sort_column_id(1, Gtk.SortType.ASCENDING)

        self.xml.get_object('LanguageSelect').set_model(self.language_list_model)
        self.xml.get_object('LanguageSelect').set_row_separator_func(self._language_list_sep)

        self.update_languages()

        parts = ['install']
        if self.install_task.bottlename:
            parts.append(self.install_task.bottlename)

        if self.install_task.profile and not self.install_task.create_new_bottle:
            parts.append(cxutils.sanitize_bottlename(self.install_task.profile.name))

        self.environment_variables = ''

        log_filename = '%s.cxlog' % '_'.join(str(x) for x in parts)

        self.logging_options = loggingoptions.LoggingOptionsController(
            log_filename, self.xml.get_object('InstallOptionsDialog'), installer=True)
        self.xml.get_object('InstallOptions').pack_start(self.logging_options.get_view(), True, False, 0)

    def run(self):
        self.xml.get_object('InstallOptionsDialog').run()

    def destroy(self):
        self.xml.get_object('InstallOptionsDialog').destroy()

    def get_cxfixes_packages(self):
        # Collect the lists of issues cxfixes can take care of
        errids = {}
        for errid in self.install_task.get_cxdiag_messages().keys():
            packages = cxfixes.get_packages(errid)
            if packages:
                for pkg in packages:
                    if pkg in errids:
                        errids[pkg].append(errid)
                    else:
                        errids[pkg] = [errid]

        # Build the package list
        packages = []
        for pkg in sorted(errids):
            if len(errids[pkg]) == 1:
                packages.append('<a href="%(url)s">%(pkg)s</a>' % {
                    'url': cxfixes.get_error_url(errids[pkg][0]),
                    'pkg': cxutils.html_escape(pkg)})
            else:
                refs = []
                for i in range(len(errids[pkg])):
                    refs.append('<a href="%(url)s">%(index)s</a>' % {
                        'url': cxfixes.get_error_url(errids[pkg][i]),
                        'index': i})
                packages.append('%s <small>[%s]</small>' % (pkg, ' '.join(refs)))

        return packages

    def update_cxfixes_packages(self):
        self.xml.get_object('InstallUnixPackagesButton')

        packages = self.get_cxfixes_packages()
        if packages:
            html = '<body><p>' + ', '.join(packages) + '</p></body>'
            self.xml.get_object('PackageNotes').display_html_safe(html)

            self.xml.get_object('InstallUnixPackagesButton').show()
            self.xml.get_object('PackageNotes').show()
        else:
            self.xml.get_object('InstallUnixPackagesButton').hide()
            self.xml.get_object('PackageNotes').hide()

    def list_dependencies(self, appid, reasons_text, reasons_rev):
        if appid not in self.install_task.profiles:
            return

        other_app_name = cxutils.html_escape(self.install_task.profiles[appid].name)
        for dep_appid in reasons_rev.get((installtask.REASON_DEPENDENCY, appid), ()):
            if dep_appid not in self.install_task.profiles:
                continue

            name = self.install_task.profiles[dep_appid].name
            if dep_appid not in self.install_task.target_bottle.installers:
                continue

            self.dependency_list_model.append((dep_appid, name, _('Required by %(other_app_name)s') % {'other_app_name': other_app_name}))

        for dep_appid in reasons_rev.get((installtask.REASON_DEPENDENCY, appid), ()):
            if dep_appid not in self.install_task.target_bottle.installers:
                continue

            self.list_dependencies(dep_appid, reasons_text, reasons_rev)

    def update_dependencies(self):
        self.dependency_list_model.clear()

        reasons_rev = {}
        if self.install_task.target_bottle and self.install_task.target_bottle.analyzed():
            for appid, reason in self.install_task.target_bottle.reasons.items():
                if reason not in reasons_rev:
                    reasons_rev[reason] = []

                reasons_rev[reason].append(appid)

            reasons_text = []
            for appid in reasons_rev.get((installtask.REASON_MAIN, None), ()):
                if appid not in self.install_task.target_bottle.installers:
                    continue

                self.dependency_list_model.append((None, self.install_task.profiles[appid].name, _('Selected for install')))
                self.list_dependencies(appid, reasons_text, reasons_rev)

            for appid in reasons_rev.get((installtask.REASON_STEAM, None), ()):
                name = self.install_task.profiles[appid].name
                if appid not in self.install_task.target_bottle.installers:
                    continue

                self.dependency_list_model.append((appid, name, _('Installing via Steam')))
                self.list_dependencies(appid, reasons_text, reasons_rev)

            for appid in reasons_rev.get((installtask.REASON_OVERRIDE, None), ()):
                if appid not in self.install_task.target_bottle.installers:
                    continue

                self.dependency_list_model.append((appid, self.install_task.profiles[appid].name, _('Manually added')))
                self.list_dependencies(appid, reasons_text, reasons_rev)

            self.xml.get_object('AddDependenciesButton').set_sensitive(True)
        else:
            self.xml.get_object('AddDependenciesButton').set_sensitive(False)

    @staticmethod
    def _language_list_cmp(model, iter1, iter2, _data):
        langid1 = model.get_value(iter1, 1)
        langid2 = model.get_value(iter2, 1)

        name1 = model.get_value(iter1, 0)
        name2 = model.get_value(iter2, 0)

        if langid1 == langid2:
            return cxutils.cmp(name1, name2)

        if langid1 == '':
            return 1

        if langid2 == '':
            return -1

        return cxutils.cmp(name1, name2)

    @staticmethod
    def _language_list_sep(model, iterator):
        langid = model.get_value(iterator, 1)
        if langid is None:
            return True

        return False

    def select_language_by_id(self, model, path, treeiter, langid):
        if model.get_value(treeiter, 1) == langid:
            self.xml.get_object("LanguageSelect").set_active(path[0])
            return True # stop iteration

        return False # continue iteration

    def update_languages(self):
        languages = self.install_task.profile_languages()
        if languages:
            self.language_list_model.clear()
            for langid in languages:
                if langid == '':
                    name = _('Other Language')
                else:
                    name = c4profiles.LANGUAGES.get(langid, langid)
                self.language_list_model.append((name, langid))

            # Add a separator
            self.language_list_model.append(('', None))

            self.xml.get_object('LanguageBox').show()
            self.language_list_model.foreach(self.select_language_by_id, self.install_task.installerLocale)
        else:
            self.xml.get_object('LanguageBox').hide()

    def on_AddDependenciesButton_clicked(self, _widget):
        if not self.xml.get_object('AddDependencyDialog').get_property('visible'):
            self.add_dependencies_list_model.clear()

            main_appid = self.install_task.profile.appid
            for appid in self.install_task.SuggestedDependencyOverrides():
                profile = self.install_task.profiles[appid]
                name = profile.name

                if main_appid in profile.app_profile.extra_fors:
                    sortkey = 'A' + name
                else:
                    sortkey = 'B' + name

                self.add_dependencies_list_model.append((appid, name, sortkey))

            add_dependencies_list_model = self.add_dependencies_list_model.filter_new()
            add_dependencies_list_model.set_visible_func(self._add_deps_filter)
            self.xml.get_object('AddDependencyList').set_model(add_dependencies_list_model)

        self.xml.get_object('AddDependencyDialog').run()
        self.xml.get_object('AddDependencyDialog').hide()

    def _add_deps_filter(self, model, iterator, _userdata):
        search_text = self.xml.get_object('AddDependencySearch').get_text()
        if search_text:
            text = model.get_value(iterator, 1)
            return search_text.lower() in text.lower()
        return True

    @staticmethod
    def on_AddDependencyDialog_delete_event(widget, _event):
        widget.hide()
        return True

    def on_AddDependencyDialog_response(self, widget, response_id):
        if response_id == 1:
            model, rows = self.xml.get_object('AddDependencyList').get_selection().get_selected_rows()
            if not rows:
                return

            appids = []
            for row in rows:
                appids.append(model.get_value(model.get_iter(row), 0))
            self.install_task.AddDependencyOverrides(appids, installtask.OVERRIDE_INCLUDE)

        widget.hide()

    def on_AddDependencyList_button_press_event(self, _widget, event):
        '''Handle double click'''
        if event.button == 1 and event.type == Gdk.EventType._2BUTTON_PRESS: # pylint: disable=W0212
            self.xml.get_object('AddDependencyDialog').response(1)

    def on_AddDependencyList_selection_changed(self, selection):
        self.xml.get_object('AddDependencyDialog').set_response_sensitive(1, selection.count_selected_rows() != 0)

    def on_AddDependencySearch_changed(self, _widget):
        self.xml.get_object('AddDependencyList').get_model().refilter()

    def on_DoneButton_clicked(self, _widget):
        self.xml.get_object('InstallOptionsDialog').close()

    def on_EnvVarEntry_changed(self, widget):
        self.environment_variables = widget.get_text()

    def on_InstallOptionsDialog_realize(self, _widget):
        if self.environment_variables:
            self.xml.get_object('EnvVarEntry').set_text(self.environment_variables)

    def on_InstallUnixPackagesButton_toggled(self, widget):
        self.install_task.apply_cxfixes = widget.get_active()

    def on_LanguageSelect_changed(self, widget):
        index = widget.get_active()
        treeiter = self.language_list_model.get_iter((index,))
        langid = self.language_list_model.get_value(treeiter, 1)
        self.install_task.SetLocale(langid)

    def on_RemoveDependenciesButton_clicked(self, _widget):
        model, rows = self.xml.get_object('DependencyList').get_selection().get_selected_rows()
        appids = []
        for row in rows:
            appid = model.get_value(model.get_iter(row), 0)
            if appid:
                appids.append(appid)

        self.install_task.AddDependencyOverrides(appids, installtask.OVERRIDE_EXCLUDE)

    def on_RevertDependenciesButton_clicked(self, _widget):
        self.install_task.ClearDependencyOverrides()

    # InstallTask delegate functions
    def profileChanged(self):
        self.update_cxfixes_packages()
        self.update_dependencies()
        self.update_languages()

    def sourceChanged(self):
        self.update_cxfixes_packages()
        self.update_dependencies()

    def analyzedBottle_(self, target_bottle):
        if target_bottle is self.install_task.target_bottle:
            self.update_cxfixes_packages()
            self.update_dependencies()

    def bottleCreateChanged(self):
        self.update_cxfixes_packages()
        self.update_dependencies()

    def bottleNewnameChanged(self):
        self.update_cxfixes_packages()
        self.update_dependencies()

    def bottleTemplateChanged(self):
        self.update_cxfixes_packages()
        self.update_dependencies()

    def bottleNameChanged(self):
        self.update_cxfixes_packages()
        self.update_dependencies()
