# (c) Copyright 2020-2021. CodeWeavers, Inc.

from gi.repository import Gtk

import cxguitools
import cxutils
import demoutils
import pyop
import webtoken

# for localization
from cxutils import cxgettext as _

class LoginController:

    def __init__(self, parent, callback=None):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("logindialog"))
        self.xml.connect_signals(self)

        self.callback = callback
        self.parent = parent

        reset_password_markup = '<small><a href="https://www.codeweavers.com/login/lostpassword/">%s</a></small>' % \
            cxutils.html_escape(_("Forgot Password?"))
        self.xml.get_object("ResetPasswordLink").set_markup(reset_password_markup)

        self.update_sensitivities()

        self.xml.get_object("EmailEntry").grab_focus()

        self.main_window = self.xml.get_object('LoginDialog')
        self.main_window.set_transient_for(parent)
        self.main_window.connect('response', self.login_clicked)
        self.main_window.show()

    def update_sensitivities(self, _caller=None):
        email_entry = self.xml.get_object("EmailEntry")
        password_entry = self.xml.get_object("PasswordEntry")
        login_button = self.xml.get_object("LoginButton")

        if password_entry.get_text() and email_entry.get_text():
            login_button.set_sensitive(True)
        else:
            login_button.set_sensitive(False)
            login_button.grab_default()

    def sensitivities_at_start_registration(self):
        email_entry = self.xml.get_object("EmailEntry")
        password_entry = self.xml.get_object("PasswordEntry")
        login_button = self.xml.get_object("LoginButton")
        cancel_button = self.xml.get_object("CancelButton")

        login_button.set_sensitive(False)
        cancel_button.set_sensitive(False)
        password_entry.set_sensitive(False)
        email_entry.set_sensitive(False)

    def sensitivities_after_registration(self):
        email_entry = self.xml.get_object("EmailEntry")
        password_entry = self.xml.get_object("PasswordEntry")
        login_button = self.xml.get_object("LoginButton")
        cancel_button = self.xml.get_object("CancelButton")

        login_button.set_sensitive(True)
        cancel_button.set_sensitive(True)
        password_entry.set_sensitive(True)
        email_entry.set_sensitive(True)

    def login_clicked(self, dialog, response):
        if response == Gtk.ResponseType.OK:
            email_entry = self.xml.get_object("EmailEntry")
            password_entry = self.xml.get_object("PasswordEntry")

            self.sensitivities_at_start_registration()
            loginOp = LoginOperation(self, email_entry.get_text(), password_entry.get_text())
            pyop.sharedOperationQueue.enqueue(loginOp)
        else:
            dialog.destroy()
            self.parent.destroy()

    def login_finished(self, success):
        self.sensitivities_after_registration()

        if success:
            self.main_window.destroy()

            if self.callback:
                self.callback()
        else:
            cxguitools.CXMessageDlg(primary=_("Login failed"),
                                    buttons=Gtk.ButtonsType.OK, parent=self.main_window,
                                    message_type=Gtk.MessageType.ERROR)

class LoginOperation(pyop.PythonOperation):

    def __init__(self, login_controller, login, password):
        pyop.PythonOperation.__init__(self)
        self.login_controller = login_controller
        self.login = login
        self.password = password
        self.success = False

    def __unicode__(self):
        return "LoginOperation with username " + self.login

    def main(self):
        self.success = webtoken.create_token(self.login, self.password)

        if not self.success:
            return

        # Also attempt to register CrossOver while we have the
        # required information. No error handling here since this is
        # not essential, and we already know that the login is valid.
        demoutils.register(self.login, self.password)

    def finish(self):
        self.login_controller.login_finished(self.success)
        pyop.PythonOperation.finish(self)
