# (c) Copyright 2020-2021 CodeWeavers, Inc.

import os
import json
import shutil
import tempfile

import urllib.request as urllib_request
import urllib.error as urllib_error
import urllib.parse as urlparse
from urllib.parse import urlencode

from gi.repository import GLib
from gi.repository import Gtk
from gi.repository import Vte

import pyop
import cxguitools

import cxhtmlutils
import cxlog
import cxproduct
import cxurlget
import cxutils
import distversion
import logindialog
import proxyinfo
import webtoken

# for localization
from cxutils import cxgettext as _

class UpdateController:

    def __init__(self, parent=None, is_reminder=False):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain("crossover")
        self.xml.add_from_file(cxguitools.get_ui_path("updater"))
        self.xml.connect_signals(self)

        self.finished = False
        self.installing = False
        self.canceled = False
        self.package_type = None
        self.is_reminder = is_reminder
        self.url = None
        self.pulse = False
        self.progress = 0

        self.xml.get_object("UpdaterDialog").set_transient_for(parent)

        if parent or not self.is_reminder:
            self.xml.get_object("UpdaterDialog").show()

        if not parent:
            self.xml.get_object("UpdaterDialog").connect('destroy', self.quit_requested)

        config = cxproduct.get_config()
        package_info = config['CrossOver'].get('ProductPackage')
        if package_info:
            self.package_type, _sep, _package_name = package_info.partition(':')

        self.get_updates()

    def get_updates(self):
        self.progress = 0
        self.update_progress()

        self.pulse = True
        GLib.timeout_add(300, self.update_progress)

        pyop.sharedOperationQueue.enqueue(GetUpdatesOperation(self.package_type, self))

    def get_updates_callback(self, update, login_required, error):
        self.progress = 1
        self.pulse = False
        self.update_progress()

        if self.canceled:
            self.destroy()

        if login_required and not self.is_reminder:
            logindialog.LoginController(self.xml.get_object("UpdaterDialog").get_toplevel(), self.get_updates)
            return

        if not self.package_type or self.package_type not in ('deb', 'rpm', 'mojo'):
            label = _('Automated updates are not supported for this type of installation.')
            update = None
        elif update:
            version = update[0]
            self.url = update[1]

            product = cxproduct.this_product()
            label = _('An update is available. Do you want to install %(product)s %(version)s now?') % {'product': product['name'], 'version': version}
        elif error:
            label = _('Failed to check for available updates.')
        else:
            label = _('No updates are available.')

        if not update:
            self.xml.get_object("CancelButton").hide()
            self.xml.get_object("OkButton").show()
        elif self.is_reminder:
            self.xml.get_object("CancelButton").hide()
            self.xml.get_object("LaterButton").show()
            self.xml.get_object("NeverButton").show()
            self.xml.get_object("InstallButton").show()
        else:
            self.xml.get_object("InstallButton").show()

        self.xml.get_object("UpdateLabel").set_text(label)

        self.xml.get_object("UpdateProgress").hide()
        self.xml.get_object("UpdaterDialog").show()

        if not update and self.is_reminder:
            self.xml.get_object("UpdaterDialog").connect('event', self.quit_requested)

    def update_progress(self):
        if self.finished:
            return False

        if self.pulse:
            self.xml.get_object("UpdateProgress").pulse()
            return True

        self.xml.get_object("UpdateProgress").set_fraction(self.progress)
        return True

    def on_later_clicked(self, _button):
        self.destroy()

    def on_never_clicked(self, _button):
        config = cxproduct.get_config()
        wconfig = config.get_save_config()
        wconfig.lock_file()
        wconfig['CrossOver']['CheckForUpdates'] = '0'
        wconfig.save_and_unlock_file()

        self.destroy()

    def on_install_clicked(self, _button):
        self.installing = True
        self.progress = 0
        self.pulse = False
        self.update_progress()

        self.xml.get_object("LaterButton").hide()
        self.xml.get_object("NeverButton").hide()
        self.xml.get_object("InstallButton").hide()
        self.xml.get_object("CancelButton").show()
        self.xml.get_object("UpdateProgress").show()

        self.xml.get_object("UpdateLabel").set_text(_('Downloading…'))

        pyop.sharedOperationQueue.enqueue(DownloadOperation(self.url, self))

    def download_callback(self, installer_file, error):
        if self.canceled:
            self.destroy()

        if error or not installer_file:
            self.xml.get_object("UpdateLabel").set_text(_('An error occurred while downloading the update.'))
            self.xml.get_object("UpdateProgress").hide()
            self.xml.get_object("CancelButton").hide()
            self.xml.get_object("OkButton").show()
            return

        self.xml.get_object("CancelButton").set_sensitive(False)
        self.xml.get_object("UpdateLabel").set_text(_('Installing…'))

        self.pulse = True

        term = Vte.Terminal()
        term.set_input_enabled(False)
        term.set_size(80, 24)

        expander = Gtk.Expander()
        expander.set_label(_("Details"))
        expander.add(term)
        expander.show_all()
        self.xml.get_object("LabelBox").add(expander)

        pyop.sharedOperationQueue.enqueue(InstallOperation(installer_file, self.package_type, term, self))

    def install_callback(self, error):
        self.progress = 1
        self.pulse = False
        self.update_progress()
        self.finished = True

        if self.canceled:
            self.destroy()

        if error:
            self.xml.get_object("UpdateLabel").set_text(_('An error occurred while installing the update.'))
            self.xml.get_object("UpdateProgress").hide()
            self.xml.get_object("CancelButton").hide()
            self.xml.get_object("OkButton").show()
            return

        product = cxproduct.this_product()
        label = _('Done. Please restart %s for the changes to take effect.' % product['name'])
        self.xml.get_object("UpdateLabel").set_text(label)

        self.xml.get_object("CancelButton").hide()
        self.xml.get_object("OkButton").show()

    def on_cancel_clicked(self, _button):
        self.canceled = True
        if not self.installing:
            self.destroy()

    def on_ok_clicked(self, _button):
        self.destroy()

    def destroy(self):
        self.finished = True
        self.xml.get_object("UpdaterDialog").destroy()

    @staticmethod
    def quit_requested(*_args):
        Gtk.main_quit()


class GetUpdatesOperation(pyop.PythonOperation):

    def __init__(self, package_type, controller):
        pyop.PythonOperation.__init__(self)
        self.controller = controller
        self.package_type = package_type
        self.update = None
        self.login_required = False
        self.error = None

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        u = UpdateUtils()
        self.update = u.get_available_updates(self.package_type)
        self.login_required = u.login_required
        self.error = u.error

    def finish(self):
        self.controller.get_updates_callback(self.update, self.login_required, self.error)
        pyop.PythonOperation.finish(self)


class DownloadOperation(pyop.PythonOperation):

    def __init__(self, url, controller):
        pyop.PythonOperation.__init__(self)
        self.controller = controller
        self.downloader = None
        self.url = url
        self.error = None

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def main(self):
        self.downloader = Downloader(self.url, self.controller)
        self.downloader.main()

        if self.downloader.error:
            self.error = self.downloader.error
            cxlog.warn(self.downloader.error)

    def finish(self):
        self.controller.download_callback(self.downloader.installer_file, self.error)
        pyop.PythonOperation.finish(self)


class InstallOperation(pyop.PythonOperation):

    def __init__(self, installer_file, package_type, term, controller):
        pyop.PythonOperation.__init__(self)
        self.controller = controller
        self.installer_file = installer_file
        self.package_type = package_type
        self.error = None
        self.term = term

    def enqueued(self):
        pyop.PythonOperation.enqueued(self)

    def spawn_cb(self, *args):
        pass

    def main(self):
        cxsu = os.path.join(cxutils.CX_ROOT, "bin", "cxsu")

        args = []
        env = None
        if self.package_type == 'deb':
            args = [cxsu, 'apt-get', 'install', self.installer_file, '-y']
        elif self.package_type == 'rpm':
            args = [cxsu, 'yum', 'localinstall', self.installer_file, '-y']
        elif self.package_type == 'mojo':
            args = [self.installer_file, '--destination', cxutils.CX_ROOT]
            if not os.access(cxutils.CX_ROOT, os.W_OK):
                args.insert(0, cxsu)

            env = os.environ.copy()
            env['CX_NOSTART'] = '1'
            env = [key + "=" + value for key, value in env.items()]
        else:
            self.error = "Package type %s not supported." % cxlog.to_str(self.package_type)
            cxlog.err(self.error)
            return

        self.term.connect('child-exited', self.callback)
        try:
            # The child_setup_data parameter is undocumented!
            self.term.spawn_async(Vte.PtyFlags.DEFAULT, None, args, env,
                                  GLib.SpawnFlags.DEFAULT,
                                  None, None, # child_setup, child_setup_data
                                  -1, # timeout
                                  None, # cancellable
                                  self.spawn_cb, None) # callback, user_data
        except AttributeError:
            # Fall back to the deprecated spawn_sync() if spawn_async() is
            # missing (which should not be the case on Vte >= 0.48 but may
            # happen anyway due to a gir parser bug).
            self.term.spawn_sync(Vte.PtyFlags.DEFAULT, None, args, env,
                                 GLib.SpawnFlags.DEFAULT,
                                 self.spawn_cb, None, # child_setup, child_setup_data
                                 None) # cancellable

    def callback(self, _vte, retcode):
        # Signal callbacks are always executed on the main thread, so this is fine.
        if retcode != 0:
            self.error = "Failed to install %s." % cxlog.to_str(self.installer_file)
            cxlog.err(self.error)

        self.controller.install_callback(self.error)

    def finish(self):
        pyop.PythonOperation.finish(self)


def get_version():
    '''Put the version number in the right format for the REST API'''
    return '.'.join(distversion.CX_VERSION.split('.')[0:3])

class UpdateUtils:

    def __init__(self):
        self.login_required = False
        self.error = None

    def get_available_updates(self, package_type):
        if not package_type:
            return None

        type_map = {'deb': 'deb',
                    'rpm': 'redhat',
                    'mojo': 'loki'}

        try:
            proxyinfo.install_default_opener()

            data = {'version': get_version(), 'token': webtoken.get_token()}
            url = 'https://www.codeweavers.com/bin/rest/downloads'
            url += '?' + urlencode(data)
            header = {'User-Agent': cxurlget.USER_AGENT}
            request = urllib_request.Request(url, None, header)

            # pylint suggests using 'with' statement, but context manager functionality
            # on the response object is undocumented
            response = urllib_request.urlopen(request, timeout=5) #pylint: disable=R1732
            data = json.loads(cxutils.string_to_unicode(response.read()))

            cxlog.log('Obtained download data:')
            cxlog.log(data)

            if not isinstance(data, dict):
                return None

            versions = sorted(data.keys(), reverse=True, key=lambda version: tuple(map(int, (version.split('.')))))
            if not versions:
                return None

            new_version = versions[0]
            return (new_version, data[new_version].get(distversion.DEMO_SKU, {}).get(type_map.get(package_type)))
        except urllib_error.HTTPError as e:
            # An error occurred while looking looking for updates, most likely
            # because the user does not have a valid token.
            if e.code == 401:
                self.login_required = True
            self.error = e.reason
            cxlog.log(e.reason)
            return None
        except urllib_error.URLError as e:
            # Errors in communication with the server.
            self.error = e.reason
            cxlog.warn(e.reason)
            return None


def get_cached_filename(url, filename):
    """Use the hashed URL so we can cache each localized installer separately,
    and so we automatically update the installer if the URL changed.
    """
    md5hasher = cxutils.md5_hasher()
    # The anchor is irrelevant to the server so strip it
    md5hasher.update(cxutils.string_to_utf8(url.split('#', 1)[0]))
    return md5hasher.hexdigest() + "." + filename

def validate_installer(filename):
    """Checks whether the specified file is an html file, and if it is,
    whether it redirects us elsewhere.
    """
    return cxhtmlutils.is_html(filename, 4096)

def get_cached_installer(url):
    """Checks whether we have an installer for the specified url in our
    downloaded installers cache.
    """
    basename = get_cached_filename(url, "")
    cxlog.log("looking for " + cxlog.to_str(basename))
    for dirpath in (cxproduct.get_installer_dir(), os.path.join(cxproduct.get_managed_dir(), "installers")):
        try:
            for dentry in os.listdir(dirpath):
                if dentry.startswith(basename):
                    path = os.path.join(dirpath, dentry)
                    if not os.path.isfile(path):
                        continue
                    cxlog.log("  -> " + cxlog.to_str(path))
                    is_html, _redirect = validate_installer(path)
                    if not is_html:
                        return path
                    # A bad file got cached somehow (maybe by an old version)
                    cxlog.log("  -> deleted (html file)")
                    os.unlink(path)
        except OSError:
            # The directory does not exist or is not readable.
            # Just skip to the next
            pass
    return None


class Downloader:

    def __init__(self, url, controller):
        self._getter = None
        self.controller = controller
        self.error = None
        self.installer_file = None
        self.url = url

    @staticmethod
    def get_installers_dir():
        dirpath = cxproduct.get_installer_dir()
        cxutils.mkdirs(dirpath)
        return dirpath

    installers_dir = property(get_installers_dir)

    def _getsize(self):
        """This is -1 if the download has not started yet.

        This is 0 if the download has started but the size is unknown.
        Otherwise it is the size in bytes.
        """
        if self._getter:
            return self._getter.bytes_total
        return -1

    size = property(_getsize)


    def _getdownloaded(self):
        """This is the number of bytes that have been downloaded."""
        if self._getter:
            return self._getter.bytes_downloaded
        return 0

    downloaded = property(_getdownloaded)


    def _getprogress(self):
        """This is a number between 0 and 1 representing the percentage that
        has been downloaded. If the download size is 0, then progress is -1.
        """
        if self._getter and self._getter.bytes_total:
            return 1.0 * self._getter.bytes_downloaded / self._getter.bytes_total
        return 0

    progress = property(_getprogress)


    def _needs_download(self):
        """Returns True if the installer needs to be downloaded, and False
        otherwise.
        """
        self.installer_file = get_cached_installer(self.url)
        if self.installer_file:
            return False

        return True

    needs_download = property(_needs_download)


    def urlgetter_failed(self, urlgetter, exception):
        if isinstance(exception, cxurlget.HttpError):
            if exception.code == 403 and urlgetter.user_agent is None:
                # In fact we expected an installer but got an HTML file
                # so we tried again with Python's default user-agent
                # (look for user_agent below) but this backfired.
                self.error = _("'%s' returned an HTML file instead of the installer") % cxlog.to_str(urlgetter.url)
            elif exception.code == 404:
                self.error = _("HTTP error 404: The file '%s' could not be found.") % cxlog.to_str(urlgetter.url)
            else:
                self.error = _("The HTTP server returned failure code %s.") % exception.code
        else:
            self.error = cxlog.debug_str(exception)

    def urlgetter_progress(self, _urlgetter):
        self.controller.progress = self.progress
        if self.controller.canceled:
            raise cxurlget.StopDownload()

    def main(self):
        if not self.needs_download or self.controller.canceled:
            return True

        installers_dir = self.get_installers_dir()
        url = self.url
        user_agent = cxurlget.USER_AGENT
        for _count in range(5):
            tmpfileno, tmppath = tempfile.mkstemp(prefix='local.tmp.', dir=installers_dir)
            # fdopen() acquires the fd as its own
            tmpfile = os.fdopen(tmpfileno, "wb")
            self._getter = cxurlget.UrlGetter(url, tmpfile, user_agent=user_agent, notify_progress=self.urlgetter_progress, notify_failed=self.urlgetter_failed)
            self._getter.fetch() # closes tmpfile for us
            if self.controller.canceled or self.error:
                os.unlink(tmppath)
                return False

            is_html, redirect = validate_installer(tmppath)
            if not is_html:
                break

            # We may have gotten an HTML file instead of the installer:
            # - We may be behind a Wi-Fi hotspot that redirects all URLs to its
            #   login page.
            # - Or the installer may be behind a web page with a <meta> refresh
            #   tag.
            if redirect is None or redirect is False:
                # Some sites (microsoft) return an HTTP 403 error when the
                # user-agent does not look like a real browser, probably to
                # ward off spiders.
                # But when given a browser-like user-agent other sites put
                # up a tracking-cookie notification page (sourceforge)
                # which requires clicking on a button (which we cannot do)
                # to get at the installer.
                # So when trying to get an installer, try first with the
                # browser-like user-agent and, if that fails, try again
                # with the default user-agent.
                if user_agent is not None:
                    # Switch to the default user-agent and try again
                    user_agent = None
                    continue
                break

            if redirect is True:
                # Just reload the url, hoping not to get a redirect again
                continue

            # Follow the web page's http-equiv redirect to make sure the
            # final destination is valid
            url = urlparse.urljoin(self._getter.url, redirect)

        if is_html:
            os.unlink(tmppath)
            self.error = _("Got an HTML file instead of an installer for %s") % self.url
            return False

        installer_file = os.path.join(installers_dir, get_cached_filename(self.url, self._getter.basename))
        try:
            shutil.move(tmppath, installer_file)
            os.chmod(installer_file, 0o777 & ~cxutils.UMASK)
        except OSError as ose:
            self.error = ose.strerror
            return False

        self.installer_file = installer_file
        return True
