#!/bin/bash

# ==== CONFIGURACIÓN (¡EDITA ESTO!) ====
PART_EFI="/dev/sdX1"    # ← ¡Cambia esto!
PART_ROOT="/dev/sdX2"   # ← ¡Y esto!
USER_NAME="nene"        # ← Tu nombre de usuario
ISO_URL="https://mirror.csclub.uwaterloo.ca/antix-linux/releases/23.2/antiX-23.2_x64-base.iso"
# =======================================

set -e  # Salir ante cualquier error

echo "⚠️  ¡ATENCIÓN! Este script formateará $PART_EFI y $PART_ROOT."
read -p "¿Continuar? (escribe 'sí' exactamente): " confirm
[[ "$confirm" != "sí" ]] && echo "Cancelado." && exit 1

ROOT_MNT="/mnt/antix"
ISO_MNT="/mnt/antix-iso"

# --- Formatear particiones ---
echo "Formateando particiones..."
sudo mkfs.fat -F32 "$PART_EFI"
sudo mkfs.ext4 -F "$PART_ROOT"

# --- Montar raíz ---
sudo mkdir -p "$ROOT_MNT"
sudo mount "$PART_ROOT" "$ROOT_MNT"

# --- Descargar y montar ISO ---
ISO_FILE="/tmp/antix-base.iso"
echo "Descargando mini-ISO de antiX..."
wget -q -O "$ISO_FILE" "$ISO_URL"

sudo mkdir -p "$ISO_MNT"
sudo mount -o loop "$ISO_FILE" "$ISO_MNT"

# --- Extraer sistema base ---
echo "Extrayendo sistema antiX..."
sudo unsquashfs -f -d "$ROOT_MNT" "$ISO_MNT/live/filesystem.squashfs"

# --- Montar EFI y sistemas virtuales ---
sudo mkdir -p "$ROOT_MNT/boot/efi"
sudo mount "$PART_EFI" "$ROOT_MNT/boot/efi"

for d in dev proc sys run; do
  sudo mount --bind "/$d" "$ROOT_MNT/$d"
done

sudo cp /etc/resolv.conf "$ROOT_MNT/etc/"

# --- Chroot: configuración automática ---
echo "Configurando sistema en chroot..."

sudo chroot "$ROOT_MNT" /bin/bash << EOF
set -e

# Actualizar y instalar dependencias EFI
apt update
DEBIAN_FRONTEND=noninteractive apt install -y grub-efi-amd64 efibootmgr

# Configurar usuario y contraseña
echo "root:root" | chpasswd
adduser --gecos "" --disabled-password "$USER_NAME"
echo "$USER_NAME:$USER_NAME" | chpasswd
usermod -aG sudo "$USER_NAME"

# Generar fstab
echo "UUID=\$(blkid -s UUID -o value $PART_ROOT) / ext4 defaults 0 1" > /etc/fstab
echo "UUID=\$(blkid -s UUID -o value $PART_EFI) /boot/efi vfat umask=0077 0 2" >> /etc/fstab

# Instalar GRUB en EFI
grub-install --target=x86_64-efi --efi-directory=/boot/efi --bootloader-id=antiX --recheck
update-grub

echo "✅ antiX base + UEFI configurado."
EOF

# --- Desmontar ---
echo "Desmontando..."
sudo umount "$ROOT_MNT/boot/efi"
for d in run sys proc dev; do
  sudo umount "$ROOT_MNT/$d"
done
sudo umount "$ROOT_MNT"
sudo umount "$ISO_MNT"

# --- Final ---
echo ""
echo "✨ ¡Listo! antiX base instalado con arranque UEFI."
echo "Particiones usadas:"
echo "  EFI: $PART_EFI"
echo "  Raíz: $PART_ROOT"
echo ""
echo "🔄 Reinicia y selecciona 'antiX' en el menú de arranque UEFI."