#!/bin/bash

# ==== CONFIGURACIÓN (¡EDITA ESTO ANTES DE EJECUTAR!) ====
PART_EFI="/dev/sdX1"    # ← ¡Reemplaza con tu partición EFI! (ej. /dev/nvme0n1p1)
PART_ROOT="/dev/sdX2"   # ← ¡Reemplaza con tu partición raíz! (ej. /dev/nvme0n1p2)
USER_NAME="nene"        # ← Nombre de tu usuario
# ========================================================

set -e

# --- Validación básica ---
[[ ! -b "$PART_EFI" ]] && echo "❌ Error: $PART_EFI no es un dispositivo de bloque." && exit 1
[[ ! -b "$PART_ROOT" ]] && echo "❌ Error: $PART_ROOT no es un dispositivo de bloque." && exit 1

# --- Obtener UUIDs en el HOST (¡clave para fstab!) ---
UUID_EFI=$(sudo blkid -s UUID -o value "$PART_EFI" 2>/dev/null || true)
UUID_ROOT=$(sudo blkid -s UUID -o value "$PART_ROOT" 2>/dev/null || true)

# Si no tienen UUID, formateamos y volvemos a leer
if [[ -z "$UUID_EFI" ]]; then
    echo "Formateando $PART_EFI como FAT32..."
    sudo mkfs.fat -F32 "$PART_EFI"
    UUID_EFI=$(sudo blkid -s UUID -o value "$PART_EFI")
fi

if [[ -z "$UUID_ROOT" ]]; then
    echo "Formateando $PART_ROOT como ext4..."
    sudo mkfs.ext4 -F "$PART_ROOT"
    UUID_ROOT=$(sudo blkid -s UUID -o value "$PART_ROOT")
fi

echo "✅ UUIDs detectadas:"
echo "  EFI: $UUID_EFI"
echo "  Raíz: $UUID_ROOT"

read -p "¿Continuar? (s/n): " -n1 -r
echo
[[ ! $REPLY =~ ^[Ss]$ ]] && echo "Cancelado." && exit 1

# --- Rutas ---
ROOT_MNT="/mnt/antix"
ISO_MNT="/mnt/antix-iso"
ISO_URL="https://mirror.csclub.uwaterloo.ca/antix-linux/releases/23.2/antiX-23.2_x64-base.iso"
ISO_FILE="/tmp/antix-23.2-base.iso"

# --- Descargar ISO ---
echo ".Descargando mini-ISO..."
wget -q --show-progress -O "$ISO_FILE" "$ISO_URL"

# --- Montar raíz ---
sudo mkdir -p "$ROOT_MNT"
sudo mount "$PART_ROOT" "$ROOT_MNT"

# --- Extraer sistema ---
echo "Extrayendo sistema base..."
sudo unsquashfs -f -d "$ROOT_MNT" <(sudo mount -o loop "$ISO_FILE" "$ISO_MNT" 2>/dev/null && cat "$ISO_MNT/live/filesystem.squashfs")

# Alternativa si el proceso anterior falla:
# sudo mount -o loop "$ISO_FILE" "$ISO_MNT"
# sudo unsquashfs -f -d "$ROOT_MNT" "$ISO_MNT/live/filesystem.squashfs"

# --- Montar EFI y pseudo-sistemas ---
sudo mkdir -p "$ROOT_MNT/boot/efi"
sudo mount "$PART_EFI" "$ROOT_MNT/boot/efi"

for d in dev proc sys run; do
    sudo mount --bind "/$d" "$ROOT_MNT/$d"
done
sudo cp /etc/resolv.conf "$ROOT_MNT/etc/"

# --- Leer contraseña segura ---
read -rsp "Introduce contraseña para '$USER_NAME': " USER_PASS
echo
read -rsp "Confirma contraseña: " USER_PASS2
echo
[[ "$USER_PASS" != "$USER_PASS2" ]] && echo "❌ Las contraseñas no coinciden." && exit 1

# --- Chroot + configuración ---
echo "Configurando sistema antiX en chroot..."

sudo chroot "$ROOT_MNT" /bin/bash << EOF_CHROOT
#!/bin/bash
set -e

apt update
DEBIAN_FRONTEND=noninteractive apt install -y grub-efi-amd64 efibootmgr

# Contraseña root (dejar sin cambiar o usar 'passwd' después)
echo "root:antix" | chpasswd

# Crear usuario
useradd -m -G sudo -s /bin/bash $USER_NAME
echo "$USER_NAME:$USER_PASS" | chpasswd

# fstab con UUIDs del HOST (ya resueltos)
cat > /etc/fstab << EOF
UUID=$UUID_ROOT / ext4 defaults 0 1
UUID=$UUID_EFI /boot/efi vfat umask=0077 0 2
EOF

# GRUB UEFI
grub-install --target=x86_64-efi --efi-directory=/boot/efi --bootloader-id=antiX --recheck
update-grub

echo "✅ Configuración completada dentro del chroot."
EOF_CHROOT

# --- Desmontar todo ---
echo "Desmontando..."
sudo umount "$ROOT_MNT/boot/efi"
for d in run sys proc dev; do
    sudo umount "$ROOT_MNT/$d"
done
sudo umount "$ROOT_MNT"
sudo umount "$ISO_MNT" 2>/dev/null || true
sudo rm -rf "$